import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "./ui/card";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Button } from "./ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Badge } from "./ui/badge";
import { Switch } from "./ui/switch";
import { ScrollArea } from "./ui/scroll-area";
import { 
  ShoppingBag, 
  Instagram, 
  Facebook, 
  MessageCircle, 
  Mail, 
  Globe,
  CheckCircle,
  AlertCircle
} from "lucide-react";
import { toast } from "sonner";

interface ChannelConfig {
  enabled: boolean;
  fields: Record<string, string>;
}

type ChannelConfigs = {
  trendyol: ChannelConfig;
  hepsiburada: ChannelConfig;
  instagram: ChannelConfig;
  facebook: ChannelConfig;
  whatsapp: ChannelConfig;
  email: ChannelConfig;
  ticimax: ChannelConfig;
};

export function Settings() {
  const [configs, setConfigs] = useState<ChannelConfigs>({
    trendyol: {
      enabled: true,
      fields: {
        storeName: "Test Mağaza",
        apiKey: "test-api-key-123",
        apiSecret: "••••••••••••",
        supplierId: "12345",
      },
    },
    hepsiburada: {
      enabled: true,
      fields: {
        storeName: "Test Mağaza HB",
        apiKey: "",
        apiSecret: "",
        merchantId: "",
      },
    },
    instagram: {
      enabled: true,
      fields: {
        username: "@testmagaza",
        accessToken: "",
        appId: "",
        appSecret: "",
      },
    },
    facebook: {
      enabled: true,
      fields: {
        pageName: "Test Mağaza",
        pageId: "",
        accessToken: "",
        appId: "",
        appSecret: "",
      },
    },
    whatsapp: {
      enabled: true,
      fields: {
        phoneNumber: "+90555 555 55 55",
        accessToken: "",
        phoneNumberId: "",
        businessAccountId: "",
      },
    },
    email: {
      enabled: true,
      fields: {
        emailAddress: "info@test.com",
        smtpHost: "smtp.gmail.com",
        smtpPort: "587",
        username: "info@test.com",
        password: "",
      },
    },
    ticimax: {
      enabled: false,
      fields: {
        storeUrl: "https://testmagaza.ticimax.com",
        apiKey: "",
        apiSecret: "",
      },
    },
  });

  const handleToggle = (channel: keyof ChannelConfigs) => {
    setConfigs({
      ...configs,
      [channel]: {
        ...configs[channel],
        enabled: !configs[channel].enabled,
      },
    });
    toast.success(
      configs[channel].enabled 
        ? `${channel} entegrasyonu devre dışı bırakıldı`
        : `${channel} entegrasyonu aktif edildi`
    );
  };

  const handleFieldChange = (
    channel: keyof ChannelConfigs,
    field: string,
    value: string
  ) => {
    setConfigs({
      ...configs,
      [channel]: {
        ...configs[channel],
        fields: {
          ...configs[channel].fields,
          [field]: value,
        },
      },
    });
  };

  const handleSave = (channel: keyof ChannelConfigs) => {
    toast.success(`${channel} ayarları kaydedildi`);
  };

  const handleTestConnection = (channel: keyof ChannelConfigs) => {
    // Simulate testing connection
    setTimeout(() => {
      toast.success(`${channel} bağlantısı başarılı!`);
    }, 1000);
  };

  return (
    <div className="h-full overflow-auto bg-gray-50">
      <div className="p-8">
        <div className="mb-8">
          <h2 className="text-3xl font-bold text-gray-900">Ayarlar</h2>
          <p className="text-gray-500 mt-1">Kanal entegrasyonlarınızı yönetin</p>
        </div>

        <Tabs defaultValue="ecommerce" className="space-y-6">
          <TabsList className="bg-white p-1">
            <TabsTrigger value="ecommerce">E-Ticaret Platformları</TabsTrigger>
            <TabsTrigger value="social">Sosyal Medya</TabsTrigger>
            <TabsTrigger value="communication">İletişim</TabsTrigger>
          </TabsList>

          {/* E-Commerce Platforms */}
          <TabsContent value="ecommerce" className="space-y-6">
            {/* Trendyol */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
                      <ShoppingBag className="h-6 w-6 text-orange-600" />
                    </div>
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        Trendyol
                        {configs.trendyol.enabled ? (
                          <Badge className="bg-green-100 text-green-700 hover:bg-green-100">
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Aktif
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            <AlertCircle className="h-3 w-3 mr-1" />
                            Pasif
                          </Badge>
                        )}
                      </CardTitle>
                      <CardDescription>Trendyol entegrasyonunu yapılandırın</CardDescription>
                    </div>
                  </div>
                  <Switch
                    checked={configs.trendyol.enabled}
                    onCheckedChange={() => handleToggle("trendyol")}
                  />
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="trendyol-store">Mağaza Adı</Label>
                    <Input
                      id="trendyol-store"
                      value={configs.trendyol.fields.storeName}
                      onChange={(e) =>
                        handleFieldChange("trendyol", "storeName", e.target.value)
                      }
                      disabled={!configs.trendyol.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="trendyol-supplier">Supplier ID</Label>
                    <Input
                      id="trendyol-supplier"
                      value={configs.trendyol.fields.supplierId}
                      onChange={(e) =>
                        handleFieldChange("trendyol", "supplierId", e.target.value)
                      }
                      disabled={!configs.trendyol.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="trendyol-key">API Key</Label>
                    <Input
                      id="trendyol-key"
                      value={configs.trendyol.fields.apiKey}
                      onChange={(e) =>
                        handleFieldChange("trendyol", "apiKey", e.target.value)
                      }
                      disabled={!configs.trendyol.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="trendyol-secret">API Secret</Label>
                    <Input
                      id="trendyol-secret"
                      type="password"
                      value={configs.trendyol.fields.apiSecret}
                      onChange={(e) =>
                        handleFieldChange("trendyol", "apiSecret", e.target.value)
                      }
                      disabled={!configs.trendyol.enabled}
                    />
                  </div>
                </div>
                <div className="flex gap-2 pt-4">
                  <Button
                    onClick={() => handleSave("trendyol")}
                    disabled={!configs.trendyol.enabled}
                  >
                    Kaydet
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => handleTestConnection("trendyol")}
                    disabled={!configs.trendyol.enabled}
                  >
                    Bağlantıyı Test Et
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* Hepsiburada */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
                      <ShoppingBag className="h-6 w-6 text-orange-600" />
                    </div>
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        Hepsiburada
                        {configs.hepsiburada.enabled ? (
                          <Badge className="bg-green-100 text-green-700 hover:bg-green-100">
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Aktif
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            <AlertCircle className="h-3 w-3 mr-1" />
                            Pasif
                          </Badge>
                        )}
                      </CardTitle>
                      <CardDescription>Hepsiburada entegrasyonunu yapılandırın</CardDescription>
                    </div>
                  </div>
                  <Switch
                    checked={configs.hepsiburada.enabled}
                    onCheckedChange={() => handleToggle("hepsiburada")}
                  />
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="hb-store">Mağaza Adı</Label>
                    <Input
                      id="hb-store"
                      value={configs.hepsiburada.fields.storeName}
                      onChange={(e) =>
                        handleFieldChange("hepsiburada", "storeName", e.target.value)
                      }
                      disabled={!configs.hepsiburada.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="hb-merchant">Merchant ID</Label>
                    <Input
                      id="hb-merchant"
                      value={configs.hepsiburada.fields.merchantId}
                      onChange={(e) =>
                        handleFieldChange("hepsiburada", "merchantId", e.target.value)
                      }
                      placeholder="12345678"
                      disabled={!configs.hepsiburada.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="hb-key">API Key</Label>
                    <Input
                      id="hb-key"
                      value={configs.hepsiburada.fields.apiKey}
                      onChange={(e) =>
                        handleFieldChange("hepsiburada", "apiKey", e.target.value)
                      }
                      placeholder="your-api-key"
                      disabled={!configs.hepsiburada.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="hb-secret">API Secret</Label>
                    <Input
                      id="hb-secret"
                      type="password"
                      value={configs.hepsiburada.fields.apiSecret}
                      onChange={(e) =>
                        handleFieldChange("hepsiburada", "apiSecret", e.target.value)
                      }
                      placeholder="your-api-secret"
                      disabled={!configs.hepsiburada.enabled}
                    />
                  </div>
                </div>
                <div className="flex gap-2 pt-4">
                  <Button
                    onClick={() => handleSave("hepsiburada")}
                    disabled={!configs.hepsiburada.enabled}
                  >
                    Kaydet
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => handleTestConnection("hepsiburada")}
                    disabled={!configs.hepsiburada.enabled}
                  >
                    Bağlantıyı Test Et
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* Ticimax */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                      <Globe className="h-6 w-6 text-blue-600" />
                    </div>
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        Ticimax
                        {configs.ticimax.enabled ? (
                          <Badge className="bg-green-100 text-green-700 hover:bg-green-100">
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Aktif
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            <AlertCircle className="h-3 w-3 mr-1" />
                            Pasif
                          </Badge>
                        )}
                      </CardTitle>
                      <CardDescription>Ticimax entegrasyonunu yapılandırın</CardDescription>
                    </div>
                  </div>
                  <Switch
                    checked={configs.ticimax.enabled}
                    onCheckedChange={() => handleToggle("ticimax")}
                  />
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2 col-span-2">
                    <Label htmlFor="ticimax-url">Mağaza URL</Label>
                    <Input
                      id="ticimax-url"
                      value={configs.ticimax.fields.storeUrl}
                      onChange={(e) =>
                        handleFieldChange("ticimax", "storeUrl", e.target.value)
                      }
                      placeholder="https://magaza.ticimax.com"
                      disabled={!configs.ticimax.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="ticimax-key">API Key</Label>
                    <Input
                      id="ticimax-key"
                      value={configs.ticimax.fields.apiKey}
                      onChange={(e) =>
                        handleFieldChange("ticimax", "apiKey", e.target.value)
                      }
                      placeholder="your-api-key"
                      disabled={!configs.ticimax.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="ticimax-secret">API Secret</Label>
                    <Input
                      id="ticimax-secret"
                      type="password"
                      value={configs.ticimax.fields.apiSecret}
                      onChange={(e) =>
                        handleFieldChange("ticimax", "apiSecret", e.target.value)
                      }
                      placeholder="your-api-secret"
                      disabled={!configs.ticimax.enabled}
                    />
                  </div>
                </div>
                <div className="flex gap-2 pt-4">
                  <Button
                    onClick={() => handleSave("ticimax")}
                    disabled={!configs.ticimax.enabled}
                  >
                    Kaydet
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => handleTestConnection("ticimax")}
                    disabled={!configs.ticimax.enabled}
                  >
                    Bağlantıyı Test Et
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Social Media */}
          <TabsContent value="social" className="space-y-6">
            {/* Instagram */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-pink-100 rounded-lg flex items-center justify-center">
                      <Instagram className="h-6 w-6 text-pink-600" />
                    </div>
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        Instagram
                        {configs.instagram.enabled ? (
                          <Badge className="bg-green-100 text-green-700 hover:bg-green-100">
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Aktif
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            <AlertCircle className="h-3 w-3 mr-1" />
                            Pasif
                          </Badge>
                        )}
                      </CardTitle>
                      <CardDescription>Instagram Business hesabınızı bağlayın</CardDescription>
                    </div>
                  </div>
                  <Switch
                    checked={configs.instagram.enabled}
                    onCheckedChange={() => handleToggle("instagram")}
                  />
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="ig-username">Kullanıcı Adı</Label>
                    <Input
                      id="ig-username"
                      value={configs.instagram.fields.username}
                      onChange={(e) =>
                        handleFieldChange("instagram", "username", e.target.value)
                      }
                      placeholder="@kullaniciadi"
                      disabled={!configs.instagram.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="ig-appid">App ID</Label>
                    <Input
                      id="ig-appid"
                      value={configs.instagram.fields.appId}
                      onChange={(e) =>
                        handleFieldChange("instagram", "appId", e.target.value)
                      }
                      placeholder="123456789"
                      disabled={!configs.instagram.enabled}
                    />
                  </div>
                  <div className="space-y-2 col-span-2">
                    <Label htmlFor="ig-token">Access Token</Label>
                    <Input
                      id="ig-token"
                      value={configs.instagram.fields.accessToken}
                      onChange={(e) =>
                        handleFieldChange("instagram", "accessToken", e.target.value)
                      }
                      placeholder="IGQVJ..."
                      disabled={!configs.instagram.enabled}
                    />
                  </div>
                  <div className="space-y-2 col-span-2">
                    <Label htmlFor="ig-secret">App Secret</Label>
                    <Input
                      id="ig-secret"
                      type="password"
                      value={configs.instagram.fields.appSecret}
                      onChange={(e) =>
                        handleFieldChange("instagram", "appSecret", e.target.value)
                      }
                      placeholder="your-app-secret"
                      disabled={!configs.instagram.enabled}
                    />
                  </div>
                </div>
                <div className="flex gap-2 pt-4">
                  <Button
                    onClick={() => handleSave("instagram")}
                    disabled={!configs.instagram.enabled}
                  >
                    Kaydet
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => handleTestConnection("instagram")}
                    disabled={!configs.instagram.enabled}
                  >
                    Bağlantıyı Test Et
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* Facebook */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                      <Facebook className="h-6 w-6 text-blue-600" />
                    </div>
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        Facebook
                        {configs.facebook.enabled ? (
                          <Badge className="bg-green-100 text-green-700 hover:bg-green-100">
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Aktif
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            <AlertCircle className="h-3 w-3 mr-1" />
                            Pasif
                          </Badge>
                        )}
                      </CardTitle>
                      <CardDescription>Facebook Page'inizi bağlayın</CardDescription>
                    </div>
                  </div>
                  <Switch
                    checked={configs.facebook.enabled}
                    onCheckedChange={() => handleToggle("facebook")}
                  />
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="fb-name">Page Adı</Label>
                    <Input
                      id="fb-name"
                      value={configs.facebook.fields.pageName}
                      onChange={(e) =>
                        handleFieldChange("facebook", "pageName", e.target.value)
                      }
                      disabled={!configs.facebook.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="fb-pageid">Page ID</Label>
                    <Input
                      id="fb-pageid"
                      value={configs.facebook.fields.pageId}
                      onChange={(e) =>
                        handleFieldChange("facebook", "pageId", e.target.value)
                      }
                      placeholder="123456789"
                      disabled={!configs.facebook.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="fb-appid">App ID</Label>
                    <Input
                      id="fb-appid"
                      value={configs.facebook.fields.appId}
                      onChange={(e) =>
                        handleFieldChange("facebook", "appId", e.target.value)
                      }
                      placeholder="123456789"
                      disabled={!configs.facebook.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="fb-secret">App Secret</Label>
                    <Input
                      id="fb-secret"
                      type="password"
                      value={configs.facebook.fields.appSecret}
                      onChange={(e) =>
                        handleFieldChange("facebook", "appSecret", e.target.value)
                      }
                      placeholder="your-app-secret"
                      disabled={!configs.facebook.enabled}
                    />
                  </div>
                  <div className="space-y-2 col-span-2">
                    <Label htmlFor="fb-token">Access Token</Label>
                    <Input
                      id="fb-token"
                      value={configs.facebook.fields.accessToken}
                      onChange={(e) =>
                        handleFieldChange("facebook", "accessToken", e.target.value)
                      }
                      placeholder="EAA..."
                      disabled={!configs.facebook.enabled}
                    />
                  </div>
                </div>
                <div className="flex gap-2 pt-4">
                  <Button
                    onClick={() => handleSave("facebook")}
                    disabled={!configs.facebook.enabled}
                  >
                    Kaydet
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => handleTestConnection("facebook")}
                    disabled={!configs.facebook.enabled}
                  >
                    Bağlantıyı Test Et
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Communication */}
          <TabsContent value="communication" className="space-y-6">
            {/* WhatsApp */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                      <MessageCircle className="h-6 w-6 text-green-600" />
                    </div>
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        WhatsApp Business
                        {configs.whatsapp.enabled ? (
                          <Badge className="bg-green-100 text-green-700 hover:bg-green-100">
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Aktif
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            <AlertCircle className="h-3 w-3 mr-1" />
                            Pasif
                          </Badge>
                        )}
                      </CardTitle>
                      <CardDescription>WhatsApp Business API entegrasyonu</CardDescription>
                    </div>
                  </div>
                  <Switch
                    checked={configs.whatsapp.enabled}
                    onCheckedChange={() => handleToggle("whatsapp")}
                  />
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="wa-phone">Telefon Numarası</Label>
                    <Input
                      id="wa-phone"
                      value={configs.whatsapp.fields.phoneNumber}
                      onChange={(e) =>
                        handleFieldChange("whatsapp", "phoneNumber", e.target.value)
                      }
                      placeholder="+90555 555 55 55"
                      disabled={!configs.whatsapp.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="wa-phoneid">Phone Number ID</Label>
                    <Input
                      id="wa-phoneid"
                      value={configs.whatsapp.fields.phoneNumberId}
                      onChange={(e) =>
                        handleFieldChange("whatsapp", "phoneNumberId", e.target.value)
                      }
                      placeholder="123456789"
                      disabled={!configs.whatsapp.enabled}
                    />
                  </div>
                  <div className="space-y-2 col-span-2">
                    <Label htmlFor="wa-token">Access Token</Label>
                    <Input
                      id="wa-token"
                      value={configs.whatsapp.fields.accessToken}
                      onChange={(e) =>
                        handleFieldChange("whatsapp", "accessToken", e.target.value)
                      }
                      placeholder="EAA..."
                      disabled={!configs.whatsapp.enabled}
                    />
                  </div>
                  <div className="space-y-2 col-span-2">
                    <Label htmlFor="wa-business">Business Account ID</Label>
                    <Input
                      id="wa-business"
                      value={configs.whatsapp.fields.businessAccountId}
                      onChange={(e) =>
                        handleFieldChange("whatsapp", "businessAccountId", e.target.value)
                      }
                      placeholder="123456789"
                      disabled={!configs.whatsapp.enabled}
                    />
                  </div>
                </div>
                <div className="flex gap-2 pt-4">
                  <Button
                    onClick={() => handleSave("whatsapp")}
                    disabled={!configs.whatsapp.enabled}
                  >
                    Kaydet
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => handleTestConnection("whatsapp")}
                    disabled={!configs.whatsapp.enabled}
                  >
                    Bağlantıyı Test Et
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* Email */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-red-100 rounded-lg flex items-center justify-center">
                      <Mail className="h-6 w-6 text-red-600" />
                    </div>
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        E-posta
                        {configs.email.enabled ? (
                          <Badge className="bg-green-100 text-green-700 hover:bg-green-100">
                            <CheckCircle className="h-3 w-3 mr-1" />
                            Aktif
                          </Badge>
                        ) : (
                          <Badge variant="secondary">
                            <AlertCircle className="h-3 w-3 mr-1" />
                            Pasif
                          </Badge>
                        )}
                      </CardTitle>
                      <CardDescription>E-posta entegrasyonunu yapılandırın</CardDescription>
                    </div>
                  </div>
                  <Switch
                    checked={configs.email.enabled}
                    onCheckedChange={() => handleToggle("email")}
                  />
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2 col-span-2">
                    <Label htmlFor="email-address">E-posta Adresi</Label>
                    <Input
                      id="email-address"
                      type="email"
                      value={configs.email.fields.emailAddress}
                      onChange={(e) =>
                        handleFieldChange("email", "emailAddress", e.target.value)
                      }
                      placeholder="info@test.com"
                      disabled={!configs.email.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="email-host">SMTP Host</Label>
                    <Input
                      id="email-host"
                      value={configs.email.fields.smtpHost}
                      onChange={(e) =>
                        handleFieldChange("email", "smtpHost", e.target.value)
                      }
                      placeholder="smtp.gmail.com"
                      disabled={!configs.email.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="email-port">SMTP Port</Label>
                    <Input
                      id="email-port"
                      value={configs.email.fields.smtpPort}
                      onChange={(e) =>
                        handleFieldChange("email", "smtpPort", e.target.value)
                      }
                      placeholder="587"
                      disabled={!configs.email.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="email-username">Kullanıcı Adı</Label>
                    <Input
                      id="email-username"
                      value={configs.email.fields.username}
                      onChange={(e) =>
                        handleFieldChange("email", "username", e.target.value)
                      }
                      placeholder="info@test.com"
                      disabled={!configs.email.enabled}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="email-password">Şifre</Label>
                    <Input
                      id="email-password"
                      type="password"
                      value={configs.email.fields.password}
                      onChange={(e) =>
                        handleFieldChange("email", "password", e.target.value)
                      }
                      placeholder="••••••••"
                      disabled={!configs.email.enabled}
                    />
                  </div>
                </div>
                <div className="flex gap-2 pt-4">
                  <Button
                    onClick={() => handleSave("email")}
                    disabled={!configs.email.enabled}
                  >
                    Kaydet
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => handleTestConnection("email")}
                    disabled={!configs.email.enabled}
                  >
                    Bağlantıyı Test Et
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}
