import { useState, useEffect, useMemo } from "react";
import { useParams } from "react-router";
import { VirtualizedMessageList } from "./VirtualizedMessageList";
import { ConversationView } from "./ConversationView";
import { mockMessages } from "../data/mockData";
import { generateLargeMessageDataset } from "../data/largeDataset";
import type { Message } from "../types";
import { Filter, Calendar } from "lucide-react";
import { Button } from "./ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "./ui/select";
import { Input } from "./ui/input";
import { Popover, PopoverContent, PopoverTrigger } from "./ui/popover";

const ITEMS_PER_PAGE = 50;

export function Messages() {
  const { channelId } = useParams();
  const [selectedMessage, setSelectedMessage] = useState<Message | null>(null);
  const [allMessages] = useState(() => generateLargeMessageDataset(5000)); // Generate 5000 messages once
  const [displayedMessages, setDisplayedMessages] = useState<Message[]>([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [isLoading, setIsLoading] = useState(false);
  const [statusFilter, setStatusFilter] = useState<"all" | "answered" | "unanswered" | "urgent">("all");
  const [channelFilter, setChannelFilter] = useState<string>("all");
  const [startDate, setStartDate] = useState<string>("");
  const [endDate, setEndDate] = useState<string>("");

  // Apply filters with useMemo
  const filteredMessages = useMemo(() => {
    let filtered = [...allMessages];

    // Channel from URL
    if (channelId) {
      filtered = filtered.filter((msg) => msg.channelType === channelId);
    }

    // Channel filter
    if (channelFilter !== "all") {
      filtered = filtered.filter((msg) => msg.channelType === channelFilter);
    }

    // Status filter
    if (statusFilter === "answered") {
      filtered = filtered.filter((msg) => msg.unread === 0);
    } else if (statusFilter === "unanswered") {
      filtered = filtered.filter((msg) => msg.unread > 0);
    } else if (statusFilter === "urgent") {
      filtered = filtered.filter((msg) => msg.isUrgent === true);
    }

    // Date range filter
    if (startDate) {
      const startDateTime = new Date(startDate).getTime();
      filtered = filtered.filter(
        (msg) => msg.timestamp.getTime() >= startDateTime
      );
    }
    if (endDate) {
      const endDateTime = new Date(endDate).setHours(23, 59, 59, 999);
      filtered = filtered.filter(
        (msg) => msg.timestamp.getTime() <= endDateTime
      );
    }

    return filtered;
  }, [allMessages, channelId, channelFilter, statusFilter, startDate, endDate]);

  // Initial load
  useEffect(() => {
    setDisplayedMessages(filteredMessages.slice(0, ITEMS_PER_PAGE));
    setCurrentPage(1);
    if (filteredMessages.length > 0 && !selectedMessage) {
      setSelectedMessage(filteredMessages[0]);
    }
  }, [filteredMessages]);

  // Load more messages
  const handleLoadMore = () => {
    if (isLoading) return;

    const nextPage = currentPage + 1;
    const startIndex = 0;
    const endIndex = nextPage * ITEMS_PER_PAGE;

    if (endIndex <= filteredMessages.length) {
      setIsLoading(true);
      
      // Simulate network delay for realism
      setTimeout(() => {
        setDisplayedMessages(filteredMessages.slice(startIndex, endIndex));
        setCurrentPage(nextPage);
        setIsLoading(false);
      }, 300);
    }
  };

  const hasMore = displayedMessages.length < filteredMessages.length;

  const handleSelectMessage = (message: Message) => {
    setSelectedMessage(message);
  };

  const handleSendMessage = (content: string) => {
    if (!selectedMessage) return;

    const newConversationMessage = {
      id: `msg-${Date.now()}`,
      sender: 'agent' as const,
      content,
      timestamp: new Date(),
    };

    // Update in both displayed and all messages
    const updateMessage = (msg: Message) =>
      msg.id === selectedMessage.id
        ? {
            ...msg,
            conversation: [...msg.conversation, newConversationMessage],
            lastMessage: content,
            timestamp: new Date(),
            unread: 0,
          }
        : msg;

    setDisplayedMessages((prev) => prev.map(updateMessage));
    setSelectedMessage((prev) =>
      prev ? updateMessage(prev) : null
    );
  };

  const handleAIResponse = (aiContent: string) => {
    // AI responses are now handled by inserting into textarea, not auto-sending
    // This function is kept for compatibility but does nothing
  };

  const handleMarkUrgent = () => {
    if (!selectedMessage) return;

    // Update message to mark as urgent
    const updateMessage = (msg: Message) =>
      msg.id === selectedMessage.id
        ? { ...msg, isUrgent: true }
        : msg;

    setDisplayedMessages((prev) => prev.map(updateMessage));
    setSelectedMessage((prev) =>
      prev ? updateMessage(prev) : null
    );
  };

  const handleRemoveUrgent = () => {
    if (!selectedMessage) return;

    // Update message to remove urgent flag
    const updateMessage = (msg: Message) =>
      msg.id === selectedMessage.id
        ? { ...msg, isUrgent: false }
        : msg;

    setDisplayedMessages((prev) => prev.map(updateMessage));
    setSelectedMessage((prev) =>
      prev ? updateMessage(prev) : null
    );
  };

  return (
    <div className="h-full flex flex-col">
      {/* Filter Bar */}
      <div className="bg-white border-b border-gray-200 p-4">
        <div className="flex items-center gap-4 flex-wrap">
          <div className="flex items-center gap-2">
            <Filter className="h-4 w-4 text-gray-500" />
            <span className="text-sm font-medium text-gray-700">Filtrele:</span>
          </div>

          <Select value={channelFilter} onValueChange={setChannelFilter}>
            <SelectTrigger className="w-48">
              <SelectValue placeholder="Kanal" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">Tüm Kanallar</SelectItem>
              <SelectItem value="trendyol-product">🛒 Trendyol - Ürün</SelectItem>
              <SelectItem value="trendyol-order">📦 Trendyol - Sipariş</SelectItem>
              <SelectItem value="hepsiburada-product">🛍️ Hepsiburada - Ürün</SelectItem>
              <SelectItem value="hepsiburada-order">📦 Hepsiburada - Sipariş</SelectItem>
              <SelectItem value="instagram-message">📸 Instagram</SelectItem>
              <SelectItem value="facebook-message">👥 Facebook</SelectItem>
              <SelectItem value="whatsapp">💬 WhatsApp</SelectItem>
              <SelectItem value="email">✉️ E-posta</SelectItem>
              <SelectItem value="ticimax">🌐 Ticimax</SelectItem>
            </SelectContent>
          </Select>

          <Select value={statusFilter} onValueChange={(value) => setStatusFilter(value as "all" | "answered" | "unanswered" | "urgent")}>
            <SelectTrigger className="w-48">
              <SelectValue placeholder="Durum" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">Tüm Mesajlar</SelectItem>
              <SelectItem value="answered">Cevaplananlar</SelectItem>
              <SelectItem value="unanswered">Cevaplanmayanlar</SelectItem>
              <SelectItem value="urgent">🚨 Acil Mesajlar</SelectItem>
            </SelectContent>
          </Select>

          <Popover>
            <PopoverTrigger asChild>
              <Button variant="outline" size="sm" className="gap-2">
                <Calendar className="h-4 w-4" />
                Tarih Aralığı
              </Button>
            </PopoverTrigger>
            <PopoverContent className="w-80">
              <div className="space-y-4">
                <div>
                  <label className="text-sm font-medium text-gray-700 mb-2 block">
                    Başlangıç Tarihi
                  </label>
                  <Input
                    type="date"
                    value={startDate}
                    onChange={(e) => setStartDate(e.target.value)}
                  />
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-700 mb-2 block">
                    Bitiş Tarihi
                  </label>
                  <Input
                    type="date"
                    value={endDate}
                    onChange={(e) => setEndDate(e.target.value)}
                  />
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  className="w-full"
                  onClick={() => {
                    setStartDate("");
                    setEndDate("");
                  }}
                >
                  Tarihleri Temizle
                </Button>
              </div>
            </PopoverContent>
          </Popover>

          {(channelFilter !== "all" || statusFilter !== "all" || startDate || endDate) && (
            <Button
              variant="ghost"
              size="sm"
              onClick={() => {
                setChannelFilter("all");
                setStatusFilter("all");
                setStartDate("");
                setEndDate("");
              }}
              className="text-blue-600 hover:text-blue-700"
            >
              Filtreleri Temizle
            </Button>
          )}
        </div>
      </div>

      <div className="flex-1 flex overflow-hidden">
        <VirtualizedMessageList
          messages={displayedMessages}
          selectedMessage={selectedMessage}
          onSelectMessage={handleSelectMessage}
          channelFilter={channelId}
          onLoadMore={handleLoadMore}
          hasMore={hasMore}
          isLoading={isLoading}
        />
        <ConversationView
          message={selectedMessage}
          onSendMessage={handleSendMessage}
          onAIResponse={handleAIResponse}
          onMarkUrgent={handleMarkUrgent}
          onRemoveUrgent={handleRemoveUrgent}
        />
      </div>
    </div>
  );
}