import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { MessageSquare, TrendingUp, Clock, CheckCircle, MessageCircle, Calendar } from "lucide-react";
import { channelStats, commentStats, mockMessages, mockComments } from "../data/mockData";
import { Link } from "react-router";
import { Badge } from "./ui/badge";
import { useState, useMemo } from "react";
import { Button } from "./ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "./ui/select";
import { Popover, PopoverContent, PopoverTrigger } from "./ui/popover";
import { Input } from "./ui/input";

type DateFilter = "today" | "yesterday" | "last7days" | "last30days" | "custom";

export function Dashboard() {
  const [dateFilter, setDateFilter] = useState<DateFilter>("today");
  const [customStartDate, setCustomStartDate] = useState<string>("");
  const [customEndDate, setCustomEndDate] = useState<string>("");

  // Filter data based on date range
  const { filteredMessages, filteredComments } = useMemo(() => {
    const now = new Date();
    let startDate: Date;
    let endDate: Date = new Date(now.setHours(23, 59, 59, 999));

    switch (dateFilter) {
      case "today":
        startDate = new Date(now.setHours(0, 0, 0, 0));
        break;
      case "yesterday":
        startDate = new Date(now.setDate(now.getDate() - 1));
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.setHours(23, 59, 59, 999));
        break;
      case "last7days":
        startDate = new Date(now.setDate(now.getDate() - 7));
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        break;
      case "last30days":
        startDate = new Date(now.setDate(now.getDate() - 30));
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        break;
      case "custom":
        if (customStartDate && customEndDate) {
          startDate = new Date(customStartDate);
          endDate = new Date(customEndDate);
          endDate.setHours(23, 59, 59, 999);
        } else {
          startDate = new Date(0);
          endDate = new Date();
        }
        break;
      default:
        startDate = new Date(now.setHours(0, 0, 0, 0));
    }

    const messages = mockMessages.filter(
      (msg) => msg.timestamp >= startDate && msg.timestamp <= endDate
    );
    const comments = mockComments.filter(
      (comment) => comment.timestamp >= startDate && comment.timestamp <= endDate
    );

    return { filteredMessages: messages, filteredComments: comments };
  }, [dateFilter, customStartDate, customEndDate]);

  const totalMessages = filteredMessages.length;
  const unreadMessages = filteredMessages.reduce((sum, msg) => sum + msg.unread, 0);
  const totalComments = filteredComments.length;
  const unreadComments = filteredComments.reduce((sum, comment) => sum + comment.unread, 0);
  const avgResponseTime = "3.5 dk";
  const resolvedToday = Math.floor(totalMessages * 0.7);

  return (
    <div className="h-full overflow-auto">
      <div className="p-8">
        <div className="mb-8 flex items-center justify-between">
          <div>
            <h2 className="text-3xl font-bold text-gray-900">Dashboard</h2>
            <p className="text-gray-500 mt-1">Mesajlarınızı ve kanallarınızı yönetin</p>
          </div>

          {/* Date Filter */}
          <div className="flex items-center gap-3">
            <Select value={dateFilter} onValueChange={(value) => setDateFilter(value as DateFilter)}>
              <SelectTrigger className="w-48">
                <SelectValue placeholder="Zaman Aralığı" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="today">📅 Bugün</SelectItem>
                <SelectItem value="yesterday">📆 Dün</SelectItem>
                <SelectItem value="last7days">📊 Son 7 Gün</SelectItem>
                <SelectItem value="last30days">📈 Son 30 Gün</SelectItem>
                <SelectItem value="custom">🗓️ Özel Tarih</SelectItem>
              </SelectContent>
            </Select>

            {dateFilter === "custom" && (
              <Popover>
                <PopoverTrigger asChild>
                  <Button variant="outline" size="sm" className="gap-2">
                    <Calendar className="h-4 w-4" />
                    Tarih Seç
                  </Button>
                </PopoverTrigger>
                <PopoverContent className="w-80">
                  <div className="space-y-4">
                    <div>
                      <label className="text-sm font-medium text-gray-700 mb-2 block">
                        Başlangıç Tarihi
                      </label>
                      <Input
                        type="date"
                        value={customStartDate}
                        onChange={(e) => setCustomStartDate(e.target.value)}
                      />
                    </div>
                    <div>
                      <label className="text-sm font-medium text-gray-700 mb-2 block">
                        Bitiş Tarihi
                      </label>
                      <Input
                        type="date"
                        value={customEndDate}
                        onChange={(e) => setCustomEndDate(e.target.value)}
                      />
                    </div>
                  </div>
                </PopoverContent>
              </Popover>
            )}
          </div>
        </div>

        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-gray-600">
                Toplam Mesaj
              </CardTitle>
              <MessageSquare className="h-4 w-4 text-gray-400" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{totalMessages}</div>
              <p className="text-xs text-gray-500 mt-1">
                {unreadMessages} okunmamış
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-gray-600">
                Toplam Yorum
              </CardTitle>
              <MessageCircle className="h-4 w-4 text-purple-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{totalComments}</div>
              <p className="text-xs text-gray-500 mt-1">
                {unreadComments} yanıtlanmamış
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-gray-600">
                Ort. Cevap Süresi
              </CardTitle>
              <Clock className="h-4 w-4 text-blue-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{avgResponseTime}</div>
              <p className="text-xs text-gray-500 mt-1">Ortalama süre</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-gray-600">
                Bugün Çözülen
              </CardTitle>
              <CheckCircle className="h-4 w-4 text-green-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{resolvedToday}</div>
              <p className="text-xs text-gray-500 mt-1">Tamamlanan</p>
            </CardContent>
          </Card>
        </div>

        {/* Channels Overview */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
          <Card>
            <CardHeader>
              <CardTitle>Mesaj Kanalları</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                {channelStats.map((channel) => (
                  <Link
                    key={channel.type}
                    to={`/messages/${channel.type}`}
                    className="flex items-center justify-between p-3 rounded-lg hover:bg-gray-50 transition-colors border border-gray-100"
                  >
                    <div className="flex items-center gap-3">
                      <div
                        className="w-10 h-10 rounded-full flex items-center justify-center"
                        style={{ backgroundColor: `${channel.color}20` }}
                      >
                        <MessageSquare
                          className="h-5 w-5"
                          style={{ color: channel.color }}
                        />
                      </div>
                      <div>
                        <div className="flex items-center gap-2">
                          <p className="font-medium text-gray-900">{channel.name}</p>
                          <Badge variant="secondary" className="text-xs">
                            {channel.category}
                          </Badge>
                        </div>
                        <p className="text-sm text-gray-500">{channel.count} mesaj</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <div
                        className="text-xl font-bold"
                        style={{ color: channel.color }}
                      >
                        {channel.count}
                      </div>
                    </div>
                  </Link>
                ))}
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Yorum Kanalları</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                {commentStats.map((channel) => (
                  <Link
                    key={channel.type}
                    to={`/comments/${channel.type}`}
                    className="flex items-center justify-between p-3 rounded-lg hover:bg-gray-50 transition-colors border border-gray-100"
                  >
                    <div className="flex items-center gap-3">
                      <div
                        className="w-10 h-10 rounded-full flex items-center justify-center"
                        style={{ backgroundColor: `${channel.color}20` }}
                      >
                        <MessageCircle
                          className="h-5 w-5"
                          style={{ color: channel.color }}
                        />
                      </div>
                      <div>
                        <div className="flex items-center gap-2">
                          <p className="font-medium text-gray-900">{channel.name}</p>
                          <Badge variant="secondary" className="text-xs">
                            {channel.category}
                          </Badge>
                        </div>
                        <p className="text-sm text-gray-500">{channel.count} yorum</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <div
                        className="text-xl font-bold"
                        style={{ color: channel.color }}
                      >
                        {channel.count}
                      </div>
                    </div>
                  </Link>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Recent Activity */}
        <Card>
          <CardHeader>
            <CardTitle>Son Aktiviteler</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {filteredMessages.slice(0, 5).length > 0 ? (
                filteredMessages.slice(0, 5).map((message) => (
                  <div
                    key={message.id}
                    className="flex items-start gap-3 p-3 rounded-lg hover:bg-gray-50"
                  >
                    <div className="w-10 h-10 rounded-full bg-gray-200 flex items-center justify-center flex-shrink-0">
                      <span className="text-sm font-semibold text-gray-600">
                        {message.customerName.charAt(0)}
                      </span>
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center justify-between">
                        <p className="text-sm font-medium text-gray-900 truncate">
                          {message.customerName}
                        </p>
                        <span className="text-xs text-gray-500 ml-2">
                          {formatTime(message.timestamp)}
                        </span>
                      </div>
                      <p className="text-sm text-gray-600 truncate">
                        {message.lastMessage}
                      </p>
                      <div className="flex items-center gap-2 mt-1">
                        <Badge variant="outline" className="text-xs">
                          {message.channelName}
                        </Badge>
                        <Badge variant="secondary" className="text-xs">
                          {message.channelCategory}
                        </Badge>
                      </div>
                    </div>
                  </div>
                ))
              ) : (
                <div className="text-center py-8 text-gray-500">
                  <p>Seçilen tarih aralığında aktivite bulunamadı.</p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}

function formatTime(date: Date): string {
  const now = new Date();
  const diff = now.getTime() - date.getTime();
  const minutes = Math.floor(diff / 60000);
  const hours = Math.floor(diff / 3600000);

  if (minutes < 60) {
    return `${minutes} dk önce`;
  } else if (hours < 24) {
    return `${hours} saat önce`;
  } else {
    return date.toLocaleDateString('tr-TR');
  }
}