import { useState, useRef, useEffect } from "react";
import { Send, Sparkles, MoreVertical } from "lucide-react";
import { Button } from "./ui/button";
import { Textarea } from "./ui/textarea";
import { Badge } from "./ui/badge";
import type { Message, ChannelType } from "../types";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "./ui/dropdown-menu";
import { toast } from "sonner";

interface ConversationViewProps {
  message: Message | null;
  onSendMessage: (content: string) => void;
  onAIResponse: (content: string) => void;
  onMarkUrgent: () => void;
  onRemoveUrgent: () => void;
}

export function ConversationView({
  message,
  onSendMessage,
  onAIResponse,
  onMarkUrgent,
  onRemoveUrgent,
}: ConversationViewProps) {
  const [inputValue, setInputValue] = useState("");
  const [isGenerating, setIsGenerating] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const textareaRef = useRef<HTMLTextAreaElement>(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" });
  };

  useEffect(() => {
    scrollToBottom();
  }, [message?.conversation]);

  const handleSend = () => {
    if (inputValue.trim() && message) {
      onSendMessage(inputValue);
      setInputValue("");
      toast.success("Mesaj gönderildi");
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleSend();
    }
  };

  const generateAIResponse = async () => {
    if (!message) return;

    setIsGenerating(true);
    toast.info("AI cevap hazırlıyor...");
    
    // Simulate AI thinking time
    await new Promise((resolve) => setTimeout(resolve, 1500));

    const lastCustomerMessage = message.conversation
      .filter((msg) => msg.sender === "customer")
      .pop();

    // Generate contextual AI responses
    const responses = {
      kargo: "Siparişinizi kontrol ettim. Kargonuz şu anda yolda ve yarın tarafınıza teslim edilecek. Kargo takip numaranız: TK123456789. Herhangi bir sorunuz olursa lütfen bize bildirin.",
      stok: "Sorduğunuz ürün önümüzdeki hafta stoklarımıza gelecek. Size geldiğinde bildirim göndermemi ister misiniz? Ayrıca benzer ürünlerimizi de inceleyebilirsiniz.",
      iade: "İade işleminiz başarıyla tamamlandı. Ücret iadesi 3-5 iş günü içerisinde banka hesabınıza yansıyacaktır. İade takip numaranız email adresinize gönderildi.",
      kampanya: "Evet, mevcut kampanyamızdan yararlanabilirsiniz! %20 indirim kuponu: WELCOME20. Bu kuponu sepet ekranında kullanabilirsiniz. Geçerlilik süresi: 7 gün.",
      default: "Mesajınız için teşekkür ederim. Konuyu detaylı inceledim ve en kısa sürede size dönüş yapacağım. Başka yardımcı olabileceğim bir konu var mı?",
    };

    let aiResponse = responses.default;
    const content = lastCustomerMessage?.content.toLowerCase() || "";

    if (content.includes("kargo") || content.includes("teslimat")) {
      aiResponse = responses.kargo;
    } else if (content.includes("stok")) {
      aiResponse = responses.stok;
    } else if (content.includes("iade")) {
      aiResponse = responses.iade;
    } else if (content.includes("kampanya") || content.includes("indirim")) {
      aiResponse = responses.kampanya;
    }

    setInputValue(aiResponse);
    setIsGenerating(false);
    toast.success("AI cevabı hazır! İnceleyip gönderin.");
  };

  const getChannelColor = (channelType: ChannelType) => {
    const colors = {
      'trendyol-product': '#F27A00',
      'trendyol-order': '#F27A00',
      'instagram-message': '#E4405F',
      'facebook-message': '#1877F2',
      'whatsapp': '#25D366',
      'email': '#EA4335',
      'ticimax': '#00A8E8',
      'hepsiburada-product': '#FF6000',
      'hepsiburada-order': '#FF6000',
    };
    return colors[channelType];
  };

  const getChannelIcon = (channelType: ChannelType) => {
    const icons = {
      'trendyol-product': '🛒',
      'trendyol-order': '📦',
      'instagram-message': '📸',
      'facebook-message': '👥',
      'whatsapp': '💬',
      'email': '✉️',
      'ticimax': '🌐',
      'hepsiburada-product': '🛍️',
      'hepsiburada-order': '📦',
    };
    return icons[channelType];
  };

  if (!message) {
    return (
      <div className="flex-1 flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <div className="w-16 h-16 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-4">
            <Send className="h-8 w-8 text-gray-400" />
          </div>
          <h3 className="text-lg font-medium text-gray-900 mb-2">
            Mesaj Seçin
          </h3>
          <p className="text-sm text-gray-500">
            Sol taraftan bir konuşma seçerek başlayın
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 flex flex-col bg-white">
      {/* Header */}
      <div className="p-4 border-b border-gray-200 flex items-center justify-between bg-gray-50">
        <div className="flex items-center gap-3">
          <div className="w-10 h-10 rounded-full bg-gray-200 flex items-center justify-center">
            <span className="text-sm font-semibold text-gray-600">
              {message.customerName.charAt(0)}
            </span>
          </div>
          <div>
            <h3 className="font-semibold text-gray-900">{message.customerName}</h3>
            <div className="flex items-center gap-2">
              <Badge
                variant="outline"
                style={{
                  borderColor: getChannelColor(message.channelType),
                  color: getChannelColor(message.channelType),
                }}
              >
                <span className="mr-1">{getChannelIcon(message.channelType)}</span>
                {message.channelName}
              </Badge>
              <Badge variant="secondary" className="text-xs">
                {message.channelCategory}
              </Badge>
              {message.isUrgent && (
                <Badge className="bg-red-500 hover:bg-red-600 text-white text-xs">
                  🚨 ACİL
                </Badge>
              )}
            </div>
          </div>
        </div>
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <button className="inline-flex items-center justify-center whitespace-nowrap rounded-md text-sm font-medium transition-colors focus-visible:outline-hidden focus-visible:ring-1 focus-visible:ring-ring disabled:pointer-events-none disabled:opacity-50 hover:bg-accent hover:text-accent-foreground h-9 w-9">
              <MoreVertical className="h-4 w-4" />
            </button>
          </DropdownMenuTrigger>
          <DropdownMenuContent align="end">
            <DropdownMenuItem 
              onClick={() => {
                if (!message.isUrgent) {
                  onMarkUrgent();
                  toast.success("Mesaj yöneticiye iletildi ve acil olarak işaretlendi!");
                } else {
                  onRemoveUrgent();
                  toast.info("Mesajın acil işaretlemesi kaldırıldı");
                }
              }}
              className={message.isUrgent ? "text-orange-600 hover:text-orange-700 focus:text-orange-700" : "text-red-600 hover:text-red-700 focus:text-red-700"}
            >
              {message.isUrgent ? "↩️ Yöneticiden Geri Çek" : "🚨 Yöneticiye İlet"}
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      </div>

      {/* Context Information */}
      {message.context && (
        <div className="border-b border-gray-200 bg-white p-4">
          {/* Ürün Bilgileri (Trendyol/Hepsiburada Ürün Soruları) */}
          {message.context.productInfo && (() => {
            // Ürün URL'ini oluştur (kanal tipine göre)
            const productUrl = message.channelType === 'trendyol-product'
              ? `https://www.trendyol.com/urun/${encodeURIComponent(message.context.productInfo!.stockCode)}`
              : `https://www.hepsiburada.com/urun/${encodeURIComponent(message.context.productInfo!.stockCode)}`;
            
            return (
              <div className="flex items-start gap-3 bg-gray-50 p-3 rounded-lg border border-gray-200">
                <img
                  src={message.context.productInfo.imageUrl}
                  alt={message.context.productInfo.name}
                  className="w-16 h-16 object-cover rounded-md border border-gray-200"
                />
                <div className="flex-1 min-w-0">
                  <a
                    href={productUrl}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="font-medium text-gray-900 text-sm line-clamp-2 hover:text-blue-600 hover:underline transition-colors"
                  >
                    {message.context.productInfo.name}
                  </a>
                  <div className="flex items-center gap-3 mt-1.5">
                    <span className="text-xs text-gray-500">
                      <span className="font-medium">Stok Kodu:</span> {message.context.productInfo.stockCode}
                    </span>
                    <span className="text-xs text-gray-500">•</span>
                    <span className="text-xs text-gray-500">
                      <span className="font-medium">Marka:</span> {message.context.productInfo.brand}
                    </span>
                  </div>
                </div>
              </div>
            );
          })()}

          {/* Sipariş Bilgileri (Trendyol/Hepsiburada Sipariş Soruları) */}
          {message.context.orderInfo && (() => {
            // Sipariş URL'ini oluştur (kanal tipine göre)
            const orderUrl = message.channelType === 'trendyol-order'
              ? 'https://www.trendyol.com/hesabim/siparislerim'
              : 'https://www.hepsiburada.com/hesabim/siparislerim';
            
            return (
              <div className="bg-gray-50 p-3 rounded-lg border border-gray-200">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-xs text-gray-500 mb-1">Sipariş No</p>
                    <a
                      href={orderUrl}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="font-semibold text-gray-900 hover:text-blue-600 hover:underline transition-colors"
                    >
                      {message.context.orderInfo.orderNumber}
                    </a>
                  </div>
                  <div className="text-right">
                    <p className="text-xs text-gray-500 mb-1">Sipariş Tarihi</p>
                    <p className="font-medium text-gray-700 text-sm">
                      {new Intl.DateTimeFormat('tr-TR', {
                        day: '2-digit',
                        month: 'long',
                        year: 'numeric',
                      }).format(message.context.orderInfo.orderDate)}
                    </p>
                  </div>
                </div>
              </div>
            );
          })()}

          {/* Talep Bilgileri (Ticimax Destek Talepleri) */}
          {message.context.ticketInfo && (
            <div className="bg-gray-50 p-3 rounded-lg border border-gray-200">
              <div className="flex items-center justify-between">
                <div className="flex-1 min-w-0 mr-4">
                  <p className="text-xs text-gray-500 mb-1">Talep Başlığı</p>
                  <p className="font-medium text-gray-900 line-clamp-1">{message.context.ticketInfo.title}</p>
                </div>
                <div className="text-right">
                  <p className="text-xs text-gray-500 mb-1">Talep Tarihi</p>
                  <p className="font-medium text-gray-700 text-sm">
                    {new Intl.DateTimeFormat('tr-TR', {
                      day: '2-digit',
                      month: 'short',
                      year: 'numeric',
                      hour: '2-digit',
                      minute: '2-digit',
                    }).format(message.context.ticketInfo.date)}
                  </p>
                </div>
              </div>
            </div>
          )}
        </div>
      )}

      {/* Messages */}
      <div className="flex-1 overflow-y-auto p-4 bg-gray-50">
        <div className="space-y-4 max-w-4xl mx-auto">
          {message.conversation.map((msg) => (
            <div
              key={msg.id}
              className={`flex ${msg.sender === "agent" ? "justify-end" : "justify-start"}`}
            >
              <div
                className={`max-w-[70%] rounded-lg p-3 shadow-sm ${
                  msg.sender === "agent"
                    ? "bg-blue-600 text-white"
                    : "bg-white text-gray-900 border border-gray-200"
                }`}
              >
                <p className="text-sm">{msg.content}</p>
                <p
                  className={`text-xs mt-1 ${
                    msg.sender === "agent" ? "text-blue-100" : "text-gray-500"
                  }`}
                >
                  {msg.timestamp.toLocaleTimeString("tr-TR", {
                    hour: "2-digit",
                    minute: "2-digit",
                  })}
                </p>
              </div>
            </div>
          ))}
          <div ref={messagesEndRef} />
        </div>
      </div>

      {/* Input Area */}
      <div className="p-4 border-t border-gray-200 bg-white">
        <div className="max-w-4xl mx-auto">
          <div className="flex gap-2 mb-3">
            <Button
              variant="outline"
              size="sm"
              onClick={generateAIResponse}
              disabled={isGenerating}
              className="flex items-center gap-2"
            >
              <Sparkles className="h-4 w-4" />
              {isGenerating ? "AI Hazırlıyor..." : "AI ile Cevap Oluştur"}
            </Button>
          </div>
          <div className="flex gap-2">
            <Textarea
              ref={textareaRef}
              placeholder="Mesajınızı yazın... (Enter ile gönderin)"
              value={inputValue}
              onChange={(e) => setInputValue(e.target.value)}
              onKeyPress={handleKeyPress}
              className="resize-none"
              rows={3}
            />
            <Button
              onClick={handleSend}
              disabled={!inputValue.trim()}
              className="flex-shrink-0 h-auto"
            >
              <Send className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}