import { Search } from "lucide-react";
import { Input } from "./ui/input";
import { Badge } from "./ui/badge";
import { ScrollArea } from "./ui/scroll-area";
import { Comment, CommentChannelType } from "../types";
import { useState } from "react";

interface CommentListProps {
  comments: Comment[];
  selectedComment: Comment | null;
  onSelectComment: (comment: Comment) => void;
  channelFilter?: string;
}

export function CommentList({
  comments,
  selectedComment,
  onSelectComment,
  channelFilter,
}: CommentListProps) {
  const [searchQuery, setSearchQuery] = useState("");

  const filteredComments = comments.filter((comment) =>
    comment.customerName.toLowerCase().includes(searchQuery.toLowerCase()) ||
    comment.comment.toLowerCase().includes(searchQuery.toLowerCase()) ||
    comment.postTitle.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const unreadCount = filteredComments.reduce((sum, comment) => sum + comment.unread, 0);

  const getChannelColor = (channelType: CommentChannelType) => {
    const colors = {
      'instagram-comment': '#E4405F',
      'facebook-comment': '#1877F2',
    };
    return colors[channelType];
  };

  const getChannelName = (channelType: CommentChannelType) => {
    const names = {
      'instagram-comment': 'Instagram',
      'facebook-comment': 'Facebook',
    };
    return names[channelType];
  };

  return (
    <div className="w-80 border-r border-gray-200 bg-white flex flex-col">
      <div className="p-4 border-b border-gray-200">
        <div className="flex items-center justify-between mb-4">
          <h3 className="font-semibold text-gray-900">
            {channelFilter ? `${getChannelName(channelFilter as CommentChannelType)} Yorumları` : 'Tüm Yorumlar'}
          </h3>
          <Badge variant="secondary">{filteredComments.length}</Badge>
        </div>
        <div className="relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
          <Input
            placeholder="Ara..."
            className="pl-10"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
          />
        </div>
      </div>

      <ScrollArea className="flex-1">
        <div className="p-2">
          {filteredComments.map((comment) => (
            <button
              key={comment.id}
              onClick={() => onSelectComment(comment)}
              className={`w-full text-left p-3 rounded-lg mb-2 transition-colors ${
                selectedComment?.id === comment.id
                  ? 'bg-blue-50 border-2 border-blue-200'
                  : 'hover:bg-gray-50 border-2 border-transparent'
              }`}
            >
              <div className="flex items-center justify-between mb-2">
                <p className="font-medium text-gray-900 text-sm">
                  @{comment.customerName}
                </p>
                <span className="text-xs text-gray-500">
                  {formatTime(comment.timestamp)}
                </span>
              </div>
              <p className="text-xs text-gray-500 mb-2 truncate">
                📌 {comment.postTitle}
              </p>
              <div className="flex items-center gap-2 flex-wrap">
                <Badge
                  variant="outline"
                  style={{ 
                    borderColor: getChannelColor(comment.channelType),
                    color: getChannelColor(comment.channelType),
                  }}
                  className="text-xs"
                >
                  {comment.channelName}
                </Badge>
              </div>
            </button>
          ))}
        </div>
      </ScrollArea>
    </div>
  );
}

function formatTime(date: Date): string {
  const now = new Date();
  const diff = now.getTime() - date.getTime();
  const minutes = Math.floor(diff / 60000);
  const hours = Math.floor(diff / 3600000);

  if (minutes < 1) {
    return 'Şimdi';
  } else if (minutes < 60) {
    return `${minutes} dk`;
  } else if (hours < 24) {
    return `${hours} saat`;
  } else {
    return date.toLocaleDateString('tr-TR', { day: 'numeric', month: 'short' });
  }
}